/************************************************************************
 * NAME:	fileio.c()
 *
 * DESCR:	file io routines for ADOS.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
#include "ados.h"
#include "standard.h"

/************************************************************************
 * NAME:	ados_sync()
 *
 * DESCR:	"Sync"s the filesystem.  This really just means to flush
 *		out the catalog and vtoc.  Note that open files are NOT
 *		flushed in this operation.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
ados_sync(struct adosfs *adosfs)
{
    if (!ados_catalog_write(adosfs)) {
	return(FALSE);
    }

    if (!ados_vtoc_write(adosfs)) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_file_delete()
 *
 * DESCR:	Deletes the given file from the filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it worked, FALSE otherwise.  If it doesn't work
 *		you should assume the file doesn't exist.  If you want
 *		to differentiate between the file not existing and the
 *		delete just not working, then check on the file existance
 *		before call this!
 *
 * NOTES:	
 ************************************************************************/
int
ados_file_delete(struct adosfs *adosfs, char *name)
{
    ados_file_entry	*file_entry;
    int			 i;
    ados_ts_ref		 location;

    file_entry = ados_catalog_lookup(adosfs,name);
    if (file_entry == NULL) {
	return(FALSE);			/* couldn't find the file	*/
    }

    /* now, to delete, need to cruise through the tslist(s) freeing	*/
    /* those sectors that are encountered...updating the allocation	*/
    /* table.  Then the catalog entry is marked as deleted.		*/

    location = file_entry->ts;

    for (; location.track != 0; ) {
	ados_tslist	*tslist;
	ados_ts_ref	*tsptr;

	tslist = ados_tslist_read(adosfs,&location);
	if (tslist == NULL) {
	    return(FALSE);
	}

	ados_tslist_enumerate(tslist); 
	while ((tsptr = ados_tslist_next(tslist)) != NULL) {
	    if (tsptr->track != 0) {			/* if sector is in use	*/
		ALLOCATION_CLEAR(adosfs->vtoc,tsptr->track,tsptr->sector);
	    }
	}

	ALLOCATION_CLEAR(adosfs->vtoc,location.track,location.sector);
	location = tslist->next;
	ados_tslist_destroy(tslist);
    }

    /* at this point all of the file storage has been unallocated	*/
    /* time to clear the catalog area itself				*/

    ados_catalog_delete(file_entry);

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_file_open()
 *
 * DESCR:	Opens a file in the given ADOS file system for either
 *		read or write.
 *		WRITE - is narrowly defined for this implementation, it
 *		actually means write-new.  Trying to WRITE over an
 *		existing file will fail.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- the incoming filename is given in "unix-style"
 *		  notation...which is translated to try to match the
 *		  ADOS notation.
 ************************************************************************/
struct ados_file *
ados_file_open(struct adosfs *adosfs, char *name)
{
    ados_file_entry	*file_entry;
    int			 i;

    /* first lookup the filename in the catalog using standard rules	*/

    file_entry = ados_catalog_lookup(adosfs,name);
    if (file_entry == NULL) {
	return(NULL);
    }

    for (i=0; i < adosfs->filecount; i++) {
	struct ados_file	*fptr = &adosfs->filebuf[i];

	if (!fptr->inuse) {
	    fptr->inuse = TRUE;
	    fptr->writemode = FALSE;
	    fptr->cursor = 0;
	    fptr->cursize = 0;
	    fptr->adosfs = adosfs;
	    fptr->entry = file_entry;
	    fptr->tslist = NULL;
	    fptr->ts = file_entry->ts;	/* first ts from the file struct	*/
	    fptr->sector_count = 0;
	    fptr->tslist_offset = 0;

	    return(fptr);
	}
    }

    return(NULL);
}


/************************************************************************
 * NAME:	hdos_file_new()
 *
 * DESCR:	Create a new file that can be written to.  It is assumed
 *		that writing will occur.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
struct ados_file *
ados_file_new(struct adosfs *adosfs, char *name)
{
    ados_file_entry	*file_entry;
    int			 i;

    file_entry = ados_catalog_lookup(adosfs,name);
    if (file_entry != NULL) {
	/* the entry already exists! */
	return(NULL);
    }

    if (!ados_catalog_new(adosfs,name)) {
	/* couldn't add it	*/
	return(NULL);
    }

    file_entry = ados_catalog_lookup(adosfs,name);
    if (file_entry == NULL) {
	/* something broke	*/
	return(NULL);
    }

    for (i=0; i < adosfs->filecount; i++) {
	struct ados_file	*fptr = &adosfs->filebuf[i];

	if (!fptr->inuse) {
	    fptr->inuse = TRUE;
	    fptr->writemode = TRUE;
	    fptr->cursor = 0;
	    fptr->cursize = 0;
	    fptr->adosfs = adosfs;
	    fptr->entry = file_entry;
	    fptr->tslist = NULL;
	    fptr->ts = file_entry->ts;	/* first ts from the file struct	*/
	    fptr->sector_count = 1;	/* always has at least one allocated	*/
	    fptr->tslist_offset = 0;

	    return(fptr);
	}
    }

    return(NULL);
}

void    
ados_file_close(struct ados_file *file)
{
    if (file->inuse) {
	if (file->writemode) {
	    ados_file_flush(file);
	    file->entry->length = file->sector_count;
	}
	file->inuse = FALSE;
    }
}

/************************************************************************
 * NAME:	ados_text_convert()
 *
 * DESCR:	Utility function for converting the "bit-high" text to
 *		"normal" text.
 *
 * NOTES:	
 ************************************************************************/
static void
ados_text_convert(char *buffer, int length)
{
    while (length--) {
	*buffer++ &= 0x7f;
    }
}

/************************************************************************
 * NAME:	ados_file_read_reload()
 *
 * DESCR:	Reloads (or initial load) the file buffer with data
 *		from the file.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the buffer was reloaded, FALSE if not.
 *
 * NOTES:	- is smart about the last sector of the buffer, will only
 *		  load up the amount of data that is valid for the given
 *		  file type.
 ************************************************************************/
static int
ados_file_read_reload(struct ados_file *file)
{
    int		 i;
    ados_ts_ref	*tsptr;

    if (file->ts.track == 0) {			/* hit EOF previously	*/
	return(FALSE);
    }

    if (file->tslist == NULL) {		/* just started on this sector	*/
	file->tslist = ados_tslist_read(file->adosfs, &(file->ts));

	if (file->tslist == NULL) {
	    return(FALSE);
	}
	file->tslist_offset = file->tslist->sector_offset;
	file->sector_count++;
	ados_tslist_enumerate(file->tslist); 
    }

#define RAW_FORMAT	TRUE
#define TEXT_CONVERT	FALSE

    while ((tsptr = ados_tslist_next(file->tslist)) != NULL) {

	if (tsptr->track != 0) {			/* if sector is in use	*/
	    if (!ados_getlsect(file->adosfs, tsptr->track, tsptr->sector, file->buffer)) {
		return(FALSE);
	    }

	    file->sector_count++;

	    file->cursize = file->bufsize;
	    file->cursor = 0;

	    {	/* process first data sector special cases	*/
		
	      if (file->sector_count == 2) {

		switch(file->entry->type) {
	            case ADOS_FILE_IBASIC:	
	            case ADOS_FILE_ABASIC:	
			file->last_bytes = int_read(&(file->buffer[0x00]));
			file->last_bytes += 2;	/* include length bytes in count	*/
			file->last_bytes %= file->bufsize;
			if (!RAW_FORMAT) {
			    file->cursor += 2;
			}
			break;

	            case ADOS_FILE_BIN:
			file->last_bytes = int_read(&(file->buffer[0x02]));
			file->last_bytes += 4;	/* include load/length bytes in count	*/
			file->last_bytes %= file->bufsize;
			if (!RAW_FORMAT) {
			    file->cursor += 4;
			}
			break;
		}
	      }
	    }

	    {	/* process the whole data buffer for some format(s)	*/

		switch(file->entry->type) {
	            case ADOS_FILE_TEXT:
			if (TEXT_CONVERT) {
			    ados_text_convert(file->buffer,file->bufsize);
			    break;
			}
		}
	    }

	    {	/* process last data block special cases */
		
	      if (file->sector_count == file->entry->length) {

		switch(file->entry->type) {
	            case ADOS_FILE_TEXT:		/* look at the block and find the \0	*/
			for (i=0; i < file->bufsize; i++) {
			    if (file->buffer[i] == '\0') {
				break;
			    }
			}
			file->cursize = i-1;
			break;

	            case ADOS_FILE_IBASIC:	
	            case ADOS_FILE_ABASIC:	
	            case ADOS_FILE_BIN:
			file->cursize = file->last_bytes;
			break;
		}
	      }
	    }

	    return(TRUE);
	}
    }
    
    /* at this point we didn't get any more sectors from the current ts_list	*/
    /* try to go on to the next tslist linked to the first one			*/
    /* it may be (track==0) meaning empty...which also indicates EOF above	*/

    file->ts = file->tslist->next;

    ados_tslist_destroy(file->tslist);
    file->tslist = NULL;

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_file_write_reload()
 *
 * DESCR:	(akin to ados_file_read_reload()) Used to "reload" the
 *		sectors so that a file can be written.  Sectors are
 *		allocated as they are needed, not AFTER.  That is, before
 *		any write call succeeds, the space is allocated for it.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if reload worked, FALSE normally means out of space.
 *
 * NOTES:	- if a FALSE occurs, the caller should back-out of the
 *		  write by freeing all of the sectors that had been
 *		  allocated.
 *		- this can be accomplished with a simple delete of the
 *		  file.
 *		- this routine, therefore, sets things up so that a
 *		  delete will always work after a failed write.
 ************************************************************************/
int
ados_file_write_reload(struct ados_file *file)
{
    int		 i;
    ados_ts_ref	*tsptr;
    ados_ts_ref	 new_sector;

    if (file->tslist == NULL) {		/* just started on this tslist	*/
	file->tslist = ados_tslist_blank(file->adosfs, file->tslist_offset);
	if (file->tslist == NULL) {
	    return(FALSE);
	}
	ados_tslist_enumerate(file->tslist); 
    }

    /* at this point, we're going to need a new sector for something	*/

    if (!ados_vtoc_sector_allocate(file->adosfs, &new_sector)) {
	return(FALSE);
    }

    file->sector_count++;

    tsptr = ados_tslist_next(file->tslist);

    if (tsptr != NULL) {
	*tsptr = new_sector;
	file->cursor = 0;
	file->cursize = file->bufsize;
	file->cursect = *tsptr;

    } else {
	/* ran out of places to scribble sectors in this tslist			*/

	file->tslist->next = new_sector;

	if (!ados_tslist_write(file->adosfs, &file->ts, file->tslist)) {
	    return(FALSE);
	}

	file->tslist_offset = file->tslist->sector_offset + file->tslist->count;
	ados_tslist_destroy(file->tslist);
	file->tslist = NULL;
	file->ts = new_sector;

	return(ados_file_write_reload(file));	/* recursively do it again	*/
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_file_read()
 *
 * DESCR:	Reads the given number of bytes from the open file.
 *
 * ARGS:	
 *
 * RETURNS:	the number of bytes read, 0 upon EOF.  -1 upon error
 *		if the number of byts returned is less than the number
 *		requested, that PROBABLY represents the last bytes in
 *		the file.  However, someone else could have written
 *		such that more bytes MAY be available upon the next call.
 *		If zero is returned, then it is definitely EOF.
 *
 * NOTES:
 ************************************************************************/
int
ados_file_read(struct ados_file *file, char *buffer, int count)
{
    int	original_count = count;
    int remaining;
    int	toxfer;

    if (!file->inuse) {
	return(-1);
    }

    if (file->writemode) {
	return(-1);
    }

    while (count) {
	remaining = file->cursize - file->cursor;

	if (remaining > 0) {
	    toxfer = MIN(remaining,count);

	    memcpy(buffer,file->buffer+file->cursor,toxfer);	/* copy what we can	*/

	    count -= toxfer;
	    file->cursor += toxfer;
	    buffer += toxfer;

	} else {

	    /* try to re/load the buffer with more data				*/
	    /* if there is none left, break and return with what we got.	*/
	    /* otherwise, continue in loop reading and reading			*/

	    if (!ados_file_read_reload(file)) {
		break;
	    }
	}
    }

    return(original_count - count);
}

/************************************************************************
 * NAME:	ados_file_write()
 *
 * DESCR:	Writes the given number of bytes to the open file.
 *
 * ARGS:	
 *
 * RETURNS:	the number of bytes written.  If the number of bytes written
 *		is less than that requested, it means that no more file
 *		space was available. -1 is returned upon error.  0 is
 *		returned if there was no more space to write anything.
 *
 * NOTES:	- 
 ************************************************************************/
int
ados_file_write(struct ados_file *file, char *buffer, int count)
{
    int	original_count = count;
    int remaining;
    int	toxfer;
    int	newgrp;

    if (!file->inuse) {
	return(-1);
    }

    if (!file->writemode) {
	return(-1);
    }

    while (count) {

	/* in the case of writing file->cursor points to the next position	*/
	/* to write to.  file->cursize is always bufsize for writing.		*/

	remaining = file->cursize - file->cursor;

	if (remaining > 0) {

	    toxfer = MIN(remaining,count);
	    memcpy(file->buffer+file->cursor,buffer,toxfer);	/* copy what we can	*/
	    count -= toxfer;
	    file->cursor += toxfer;
	    buffer += toxfer;

	} else {

	    /* at this point we ran out of space in the current buffer (sector)	*/
	    /* and need to find more space.  But first, write out the existing	*/
	    /* buffer if there is something to write out.  Note that this flush	*/
	    /* also writes out the current tslist...which is probably not necessary. */

	    if (!ados_file_flush(file)) {
		return(FALSE);
	    }


	    if (!ados_file_write_reload(file)) {
		return(0);	/* a return of 0 will already have flushed the file	*/
	    }
	}
    }

    return(original_count - count);
}

/************************************************************************
 * NAME:	ados_file_flush()
 *
 * DESCR:	Flush the current file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 ************************************************************************/
int
ados_file_flush(struct ados_file *file)
{
    ados_ts_ref	*tsptr;
    int		 i;


    /* assumes that the catalog/vtoc will be written	*/
    /* when a sync is done.				*/

    /* first write out the current sector		*/

    if (file->cursize != 0) {

	/* un-written positions within a sector are set to zero	*/
	/* this is important for TEXT file types for sure	*/

	for( i=file->cursor; i < file->bufsize; i++) {
	    file->buffer[i] = '\0';
	}

	if (!ados_putlsect(file->adosfs, file->cursect.track, file->cursect.sector, file->buffer)) {	
	    return(FALSE);
	}

	/* then write out the current tslist if we can	*/

	if ( file->tslist != NULL) {
	    if (!ados_tslist_write(file->adosfs, &file->ts, file->tslist)) {
		return(FALSE);
	    }
	}
    }

    return(TRUE);
}


/************************************************************************
 * NAME:	ados_fileio_init()
 *
 * DESCR:	Allocate the file_handle structure within the adosfs with the
 *		given count (max open files).
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if done ok, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
ados_fileio_init(struct adosfs *adosfs, int count)
{
    int		i;

    adosfs->filecount = 0;

    adosfs->filebuf = (struct ados_file *)malloc(count * sizeof(struct ados_file));
    if (adosfs->filebuf == NULL) {
	return(FALSE);
    }

    adosfs->filecount = count;
    for (i=0; i < count; i++) {
	adosfs->filebuf[i].adosfs = adosfs;
	adosfs->filebuf[i].inuse = FALSE;
	adosfs->filebuf[i].bufsize = adosfs->vtoc->sector_size;
	adosfs->filebuf[i].buffer = (unsigned char *)malloc(adosfs->filebuf[i].bufsize);
	if (adosfs->filebuf[i].buffer == NULL) {
	    free(adosfs->filebuf);
	    return(FALSE);
	}
    }
    return(TRUE);
}

void
ados_fileio_cleanup(struct adosfs *adosfs)
{
    int	i;

    for (i=0; i < adosfs->filecount; i++) {
	free(adosfs->filebuf[i].buffer);
    }

    free(adosfs->filebuf);
}

/************************************************************************
 * NAME:	ados_file_stat()
 *
 * DESCR:	Returns a stat structure filled in with the stats of
 *		the given file handle.  Assumes that the file has been
 *		open in one fashion or another.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if things went OK, FALSE otherwise
 *
 * NOTES:	- fills in the given stat buffer
 ************************************************************************/
int
ados_file_stat(struct ados_file *file, struct ados_statbuf *statbuf)
{
    ados_ts_ref		 location, last, first;
    ados_tslist		*tslist;

    last.track = 0;
    first.track = 0;

    if (!file->inuse) {
	return(FALSE);
    }

    statbuf->type = file->entry->type;
    statbuf->ts_blocks = 0;
    statbuf->blocks = 0;
    statbuf->length = 0;
    statbuf->start = 0;

    location = file->ts;
    while (location.track != 0) {
	tslist = ados_tslist_read(file->adosfs, &location);
	if (tslist == NULL) {
	    return(FALSE);
	}

	statbuf->ts_blocks++;

	{	/* count the number of blocks in this tslist	*/

	    int	i;
	    for (i=0; i < tslist->count; i++) {
		if (tslist->list[i].track != 0) {
		    if (first.track == 0) {
			first = tslist->list[i];	/* keep track of first filled sector	*/
		    }
		    statbuf->blocks++;
		    last = tslist->list[i];		/* keep track of last filled sector	*/
		}
	    }
	}

	location = tslist->next;
	ados_tslist_destroy(tslist);
    }

    {	/* figure out the actual size of the file	*/

	unsigned char	*data;
	ados_ts_ref	target;
	int		i;

	data = (unsigned char *)malloc(file->adosfs->vtoc->sector_size);
	if (data == NULL) {
	    return(FALSE);
	}


	switch(file->entry->type) {
	    case ADOS_FILE_TEXT:		/* look at the last block and find the \0	*/
		target = last;
		break;

	    case ADOS_FILE_IBASIC:	/* look at the first block, first word for length	*/
	    case ADOS_FILE_ABASIC:	/* look at the first block, first word for length	*/
	    case ADOS_FILE_BIN:		/* look at the first block, second word for length	*/
		target = first;
		break;

	    case ADOS_FILE_S:		/* these types have no characteristics beyond blocks	*/
	    case ADOS_FILE_R:
	    case ADOS_FILE_A:
	    case ADOS_FILE_B:
		target.track = 0;
		break;
	}

	if (target.track != 0) {
	    if (!ados_getlsect(file->adosfs, target.track, target.sector, data)) {
		free(data);
		return(FALSE);
	    }
	}

	switch(file->entry->type) {
	    case ADOS_FILE_TEXT:		/* look at the last block and find the \0	*/
		statbuf->length = file->adosfs->vtoc->sector_size * (statbuf->blocks-1);
		for (i=0; i < file->adosfs->vtoc->sector_size; i++) {
		    if (data[i] == '\0') {
			break;
		    }
		}
		statbuf->length += i;
		break;


	    case ADOS_FILE_IBASIC:	/* look at the first block, first word for length	*/
	    case ADOS_FILE_ABASIC:	/* look at the first block, first word for length	*/
		statbuf->length = int_read(&(data[0x00]));
		break;

	    case ADOS_FILE_BIN:		/* look at the first block, second word for length	*/
	                        /* first word is the loading address			*/
		statbuf->start = int_read(&(data[0x00]));
		statbuf->length = int_read(&(data[0x02]));
		break;

	    case ADOS_FILE_S:		/* these types have no characteristics beyond blocks	*/
	    case ADOS_FILE_R:
	    case ADOS_FILE_A:
	    case ADOS_FILE_B:
		statbuf->length = file->adosfs->vtoc->sector_size * statbuf->blocks;
		break;
	}

	free(data);
    }

    return(TRUE);
}
